<?php
/**
 * Created by PhpStorm.
 * User: workclem
 * Date: 10/07/17
 * Time: 16:29
 */

namespace Project\Managers;

use GuzzleHttp\Client;
use ObjectivePHP\Application\Session\Session;
use ObjectivePHP\Notification\Alert;
use ObjectivePHP\Notification\Success;
use ObjectivePHP\ServicesFactory\Annotation\Inject;
use ObjectivePHP\ServicesFactory\Specs\InjectionAnnotationProvider;
use Project\Entity\BlogPost;
use Project\Entity\Comment;
use Project\Hydrators\BlogPostHydrator;
use Zend\Cache\Storage\StorageInterface;

class BlogPostManager extends AbstractManager implements InjectionAnnotationProvider
{
    /**
     * @var Client
     * @Inject(service="http.client")
     */
    protected $client;

    /**
     * @var BlogPostHydrator
     * @Inject(service="blog-post.hydrator")
     */
    protected $blogPostHydrator;

    /**
     * @return StorageInterface
     * @Inject(service="cache")
     */
    protected $cache;

    /**
     * @var int
     * @Inject(param="posts-per-page")
     */
    protected $nbPerPage;

    public function fetchAllBlogPost()
    {
        if (!$results = $this->getCache()->getItem('blog')) {
            $blogPosts = $this->query('blog');
            $results = [];
            foreach ($blogPosts as $post) {
                //@Todo we may want to reduce the amount of http queries
                $post['comments'] = $this->query('comments', ['post' => $post['id']]);
                array_push($results, $this->getBlogPostHydrator()->hydrate(new BlogPost(), $post));
            }
            $this->getCache()->setItem('blog', serialize($results));
        } else {
            $results = unserialize($results);
        }

        return $results;
    }

    public function getOnePost($page = 1)
    {
        $page = (int)$page;
        if ($page < 1) {
            $page = 1;
        }
        $nbPages = $this->countPages();
        if ($page > $nbPages) {
            $page = $nbPages;
        }
        $results = $this->fetchAllBlogPost();
        $results = array_chunk($results, $this->nbPerPage);

        return $results[$page - 1] ?? [];
    }

    public function countPages()
    {
        $results = $this->fetchAllBlogPost();
        $nbPages = (int)ceil(count($results) / $this->nbPerPage);

        return $nbPages === 0 ? 1 : $nbPages;
    }

    public function addComment(Comment $comment)
    {
        try {
            $response = $this->client->request('POST', 'comments', [
                'form_params' => [
                    'author_name' => $comment->getAuthorName(),
                    'author_email' => $comment->getAuthorEmail(),
                    'content' => $comment->getContent(),
                    'post' => $comment->getPost()
                ]
            ]);
            $response = json_decode($response->getBody(), true);

            if (isset($response['id'])) {
                (new Session('notifications'))
                    ->set(
                        'success',
                        new Success(
                            'Une fois validé par l\'administrateur votre commentaire sera visible.<br>
                            Merci de votre contribution'
                        )
                    );
            }
        } catch (\Exception $e) {
            (new Session('notifications'))
                ->set(
                    'danger',
                    new Alert(
                        'Une erreur est survenue pendant l\'ajout du commentaire<br>
                        (' . $e->getMessage() . ')'
                    )
                );
        }
    }

    /**
     * @return BlogPostHydrator
     */
    public function getBlogPostHydrator(): BlogPostHydrator
    {
        return $this->blogPostHydrator;
    }

    /**
     * @param BlogPostHydrator $blogPostHydrator
     */
    public function setBlogPostHydrator(BlogPostHydrator $blogPostHydrator)
    {
        $this->blogPostHydrator = $blogPostHydrator;
    }

    /**
     * @return mixed
     */
    public function getCache()
    {
        return $this->cache;
    }

    /**
     * @param mixed $cache
     */
    public function setCache($cache)
    {
        $this->cache = $cache;
    }
}
